const Joi = require("@hapi/joi");

const vector3Schema = Joi.object({
    x: Joi.number().required(),
    y: Joi.number().required(),
    z: Joi.number().required()
});

const vector2Schema = Joi.object({
    x: Joi.number().required(),
    y: Joi.number().required()
});

const laneIdSchema = Joi.object({
    StartNode: Joi.number(),
    EndNode: Joi.number(),
    Type: Joi.number()
}).allow(null);

const timeStamp = Joi.number().min(0).required();
const mapName = Joi.string();
const navigationMethod = Joi.string();

const logEvent = Joi.object({
    timeStamp: timeStamp,
    '$type': Joi.string().required(),
});

const logEvents = {
    offRoad: logEvent.append({
        type: "OffRoadEvent",

        startTime: timeStamp,
        endTime: timeStamp,
    }),

    position: logEvent.append({
        type: "PositionEvent",

        position: vector3Schema,
        rotation: Joi.number().min(-180).max(180),
        velocity: Joi.number().min(0),
        lookDirection: vector3Schema,
        headPosition: vector3Schema,
        steering: Joi.number().min(-1).max(1),
        acceleration: Joi.number().min(0).max(1),
        braking: Joi.number().min(0).max(1),
        signaling: Joi.number().min(-1).max(2),
        drive: Joi.number().min(-1).max(1),
        intersectionId: Joi.number().integer(),
        laneId: laneIdSchema,
        lane: Joi.number().min(0),
        distanceToLaneCenter: Joi.alternatives().try(Joi.number(), ("NaN")),
        speed: Joi.number(),
        relativeSpeed: Joi.number(),
        speedLimit: Joi.number(),
        naviRouteState: Joi.number().integer().min(-1).max(1),
    }),

    collision: logEvent.append({
        type: "CollisionEvent",

        objectType: Joi.string().allow(''),
        objectName: Joi.string().allow(''),
    }),

    consents: logEvent.append({
        type: "ConsentsEvent",

        consents: Joi.bool(),
    }),

    inputTypeChange: logEvent.append({
        type: "InputTypeChangeEvent",

        inputDeviceName: Joi.string().allow(''),
        inputValue: Joi.string().allow('')
    }),

    start: logEvent.append({
        type: "StartEvent",
        inputDevice: Joi.string(),
        outputDevice: Joi.string(),
        participantId: Joi.string().allow(''),
        evaluationVariationId: Joi.string().allow(''),
        isDebug: Joi.bool()
    }),

    stop: logEvent.append({
        type: "StopEvent",
    }),

    enterMap: logEvent.append({
        type: "EnterMapEvent",

        mapName: mapName,
        navigationMethod: navigationMethod,
    }),

    exitMap: logEvent.append({
        type: "ExitMapEvent",

        mapName: mapName,
        navigationMethod: navigationMethod,
        mapErrorCount: Joi.number().integer().min(0)
    }),

    naviWaypointReached: logEvent.append({
        type: "NaviWaypointReachedEvent",

        waypointIndex: Joi.number().integer().min(0),
        waypointPosition: vector2Schema,
    }),

    signalingEvent: logEvent.append({
        type: "SignalingEvent",

        signaling: Joi.bool(),
        correctSignaling: Joi.bool(),
        distanceToWaypoint: Joi.number(),
        speed: Joi.number(),
        angleToWaypoint: Joi.number(),
    }),

    demographicQuestionnaireEvent: logEvent.append({
        type: "DemographicQuestionnaireEvent",
        age: Joi.number().min(14).allow(null),
        gender: Joi.string().allow(null),
        country: Joi.string().allow(null),
        drivingEnvironment: Joi.array().items(Joi.string()).allow(null),
        accidents: Joi.number().min(0).max(11).allow(null),
        driversLicense: Joi.string().allow(null),
        driversLicenseTypes: Joi.array().items(Joi.string()).allow(null),
        drivingHabits: Joi.string().allow(null),
        drivingHabitsWithNavi: Joi.string().allow(null),
        naviUseCases: Joi.array().items(Joi.string()).allow(null),
        kindOfNavi: Joi.array().items(Joi.string()).allow(null),
        vrTime: Joi.string().allow(null),
        vrUsage: Joi.array().items(Joi.string()).allow(null),
        vrTypes: Joi.array().items(Joi.string()).allow(null),
        colorVision: Joi.string().allow(null),
        simSickness: Joi.string().allow(null),
    }),

    postQuestionnaireEvent: logEvent.append({
        type: "PostQuestionnaireEvent",
        naviUsabilityFirst: Joi.string().allow(null),
        naviUsabilityLast: Joi.string().allow(null),
    }),

    afterMapQuestionnaireEvent: logEvent.append({
        type: "AfterMapQuestionnaireEvent",
        questionnaireType: Joi.string(),
        answers: Joi.array().items(Joi.number().integer()),
        mapName: mapName,
        navigationMethod: navigationMethod
    }),

    teleportEvent: logEvent.append({
        type: "TeleportEvent",
        teleportReason: Joi.string(),
    }),

    statsAccumulatorEvent: logEvent.append({
        type: "StatsAccumulatorEvent",
        accelerationArithmeticMean: Joi.alternatives().try(Joi.number(), ("NaN")),
        accelerationMax: Joi.alternatives().try(Joi.number(), ("NaN")),
        brakingArithmeticMean: Joi.alternatives().try(Joi.number(), ("NaN")),
        brakingMax: Joi.alternatives().try(Joi.number(), ("NaN")),

        signalingTimeMean: Joi.alternatives().try(Joi.number(), ("NaN")),
        signalingTimeMin: Joi.alternatives().try(Joi.number(), ("NaN")),
        signalingTimeMax: Joi.alternatives().try(Joi.number(), ("NaN")),
        signalingTime10Percentile: Joi.alternatives().try(Joi.number(), ("NaN")),
        signalingTime90Percentile: Joi.alternatives().try(Joi.number(), ("NaN")),

        distanceToLaneCenterMean: Joi.alternatives().try(Joi.number(), ("NaN")),
        distanceToLaneCenterMin: Joi.alternatives().try(Joi.number(), ("NaN")),
        distanceToLaneCenterMax: Joi.alternatives().try(Joi.number(), ("NaN")),
        distanceToLaneCenter10Percentile: Joi.alternatives().try(Joi.number(), ("NaN")),
        distanceToLaneCenter90Percentile: Joi.alternatives().try(Joi.number(), ("NaN")),

        notOnNaviRouteCount: Joi.number().integer().min(0),
        notOnNaviRouteTimeTotal: Joi.alternatives().try(Joi.number(), ("NaN")),
        notOnNaviRouteTimeMean: Joi.alternatives().try(Joi.number(), ("NaN")),
        notOnNaviRouteTimeMin: Joi.alternatives().try(Joi.number(), ("NaN")),
        notOnNaviRouteTimeMax: Joi.alternatives().try(Joi.number(), ("NaN")),
        notOnNaviRouteTime10Percentile: Joi.alternatives().try(Joi.number(), ("NaN")),
        notOnNaviRouteTime90Percentile: Joi.alternatives().try(Joi.number(), ("NaN")),

        meanSpeedPerSpeedLimit: Joi.any(),
        meanRelativeSpeedPerSpeedLimit: Joi.any(),

        mapName: Joi.string(),
        naviType: Joi.string(),

        totalTime: Joi.number(),
        errorCount: Joi.number().integer().min(0),
    }),

    stopSignEvent: logEvent.append({
        type: "StopSignEvent",
        intersectionId: Joi.number().min(1),
        laneId: laneIdSchema,
        distanceToIntersection: Joi.alternatives().try(Joi.number(), ("NaN")),
        hasStopped: Joi.bool(),
        enterSpeed: Joi.number().min(0),
    }),

    lookBeforeEnterIntersectionEvent: logEvent.append({
        type: "LookBeforeEnterIntersectionEvent",
        intersectionId: Joi.number().min(1),
        intersectionPriority: Joi.string(),
        laneId: laneIdSchema,
        enterTime: timeStamp,
        timeLookLeft: Joi.number().min(0),
        timeLookRight: Joi.number().min(0),
        timeLookLeftSecond: Joi.number().min(0),
        lookedLeftRightLeftInTimer: Joi.bool(),
        lookedInTimeWindow: Joi.bool()
    })
}

const logsSchema = Joi.array()
    .items(
        logEvents.consents,
        logEvents.start,
        logEvents.stop,
        logEvents.demographicQuestionnaireEvent,
        logEvents.afterMapQuestionnaireEvent,
        logEvents.enterMap,
        logEvents.exitMap,
        logEvents.statsAccumulatorEvent,
        logEvents.postQuestionnaireEvent,

        logEvents.collision,
        logEvents.teleportEvent,
        logEvents.signalingEvent,
        logEvents.inputTypeChange,
    )
    .default([]);


const mapLogSchema = Joi.array()
    .items(
        logEvents.offRoad,
        logEvents.position,
        logEvents.inputTypeChange,
        logEvents.naviWaypointReached,
        logEvents.teleportEvent,
        logEvents.collision,
        logEvents.signalingEvent,
        logEvents.stopSignEvent,
        logEvents.lookBeforeEnterIntersectionEvent,
    )
    .default([]);

module.exports = {
    logsSchema,
    mapLogSchema
}